<?php
/**
 * Custom template tags for this theme
 *
 * @package Nur
 */

if ( ! function_exists( 'nur_site_branding_text' ) ) :
	/**
	 * Display Site Title and Tagline.
	 */
	function nur_site_branding_text() {
		?>
		<div class="site-branding-text">
			<?php if ( is_front_page() && is_home() ) : ?>
				<h1 class="site-title"><a href="<?php echo esc_url( home_url( '/' ) ); ?>" rel="home"><?php bloginfo( 'name' ); ?></a></h1>
			<?php else : ?>
				<p class="site-title"><a href="<?php echo esc_url( home_url( '/' ) ); ?>" rel="home"><?php bloginfo( 'name' ); ?></a></p>
			<?php endif; ?>

			<?php
			$description = get_bloginfo( 'description', 'display' );
			if ( ( ! is_front_page() || ! is_home() || ! nur_show_page_header() ) && '' !== $description ) :
				?>
				<p class="site-description"><?php echo esc_html( $description ); ?></p>
			<?php endif; ?>
		</div>
		<?php
	}
endif;

if ( ! function_exists( 'nur_header_image_tag' ) ) :
	/**
	 * Display image tag markup for a custom header image.
	 */
	function nur_header_image_tag() {
		$image_id = false;

		if ( is_singular() ) {
			$image_id = get_post_meta( get_the_ID(), '_nur_header_image_id', true );
		}

		if ( ! $image_id && nur_is_blog() ) {
			$image_id = get_theme_mod( 'blog_header_image_id' );
		}

		$attr = nur_get_header_image_attr( $image_id );
		the_header_image_tag( $attr );
	}
endif;

if ( ! function_exists( 'nur_the_page_title' ) ) :
	/**
	 * Display the current page or post title.
	 */
	function nur_the_page_title() {
		// 404 page
		if ( is_404() ) {
			esc_html_e( 'Page not found', 'nur' );
			return;
		}
		// Search pages
		if ( is_search() ) {
			/* translators: %s is the search query */
			printf( esc_html__( 'Search Results for "%s"', 'nur' ), esc_html( get_search_query() ) );
			return;
		}
		// WooCommerce pages
		if ( nur_is_woocommerce() ) {
			woocommerce_page_title();
			return;
		}
		// Archive pages
		if ( is_archive() ) {
			the_archive_title();
			return;
		}
		// By default show the title
		single_post_title();
	}
endif;

if ( ! function_exists( 'nur_breadcrumbs' ) ) :
	/**
	 * Display the breadcrumbs trail.
	 *
	 * Depends on Breadcrumb NavXT plugin being installed.
	 */
	function nur_breadcrumbs() {
		$trail = bcn_display( true );
		if ( ! nur_show_breadcrumbs() || ! $trail ) {
			return;
		}
		?>
		<div class="breadcrumbs" typeof="BreadcrumbList" vocab="http://schema.org/">
			<?php echo wp_kses( nur_bcn_append_separator( $trail ), 'breadcrumb_navxt' ); ?>
		</div>
		<?php
	}
endif;

if ( ! function_exists( 'nur_cart_link' ) ) :
	/**
	 * Displays the cart link.
	 *
	 * Depends on WooCommerce plugin being installed.
	 */
	function nur_cart_link() {
		?>
		<a class="cart-contents" href="<?php echo esc_url( wc_get_cart_url() ); ?>" title="<?php esc_attr_e( 'View your shopping cart', 'nur' ); ?>">
			<?php
			$count = WC()->cart->get_cart_contents_count();
			if ( $count ) :
				?>
				<span class="count"><?php echo esc_html( $count ); ?></span>
			<?php endif; ?>
		</a>
		<?php
	}
endif;

if ( ! function_exists( 'nur_pagination' ) ) :
	/**
	 * Displays the paging navigation on archive pages.
	 */
	function nur_pagination() {
		the_posts_pagination( array(
			'prev_text'          => '<i class="typcn typcn-arrow-left"></i><span class="screen-reader-text">' . esc_html__( 'Previous page', 'nur' ) . '</span>',
			'next_text'          => '<i class="typcn typcn-arrow-right"></i><span class="screen-reader-text">' . esc_html__( 'Next page', 'nur' ) . '</span>',
			'before_page_number' => '<span class="meta-nav screen-reader-text">' . esc_html__( 'Page', 'nur' ) . ' </span>',
		) );
	}
endif;

if ( ! function_exists( 'nur_post_meta' ) ) :
	/**
	 * Displays meta information related to the post.
	 */
	function nur_post_meta() {
		?>
		<footer class="entry-meta">

			<?php if ( is_sticky() && is_home() ) : ?>
				<span class="entry-sticky-badge"><span class="screen-reader-text"><?php echo esc_html__( 'Featured post', 'nur' ); ?></span></span>
			<?php endif; ?>

			<time class="entry-date" datetime="<?php echo esc_attr( get_the_time( 'c' ) ); ?>">
				<a href="<?php the_permalink(); ?>" rel="bookmark"><?php echo esc_html( get_the_date() ); ?></a>
			</time>

			<?php if ( is_singular( 'post' ) ) : ?>
				&nbsp;&nbsp;/&nbsp;&nbsp;
				<?php
				/* translators: %s is the post author */
				printf( esc_html__( 'By %s', 'nur' ), wp_kses_post( get_the_author_posts_link() ) );
				?>
			<?php endif; ?>

			<?php
			/* translators: used between list items, there is a space after the comma */
			$categories = get_the_category_list( __( ', ', 'nur' ) );
			if ( $categories ) :
				?>
				&nbsp;&nbsp;/&nbsp;&nbsp; <?php echo wp_kses_post( $categories ); ?>
			<?php endif; ?>

			<?php
			/* translators: used between list items, there is a space after the comma */
			$tags = get_the_tag_list( '', __( ', ', 'nur' ) );
			if ( $tags ) :
				?>
				&nbsp;&nbsp;/&nbsp;&nbsp;
				<?php
				/* translators: %s is a list of tags */
				printf( esc_html__( 'Tags: %s', 'nur' ), wp_kses_post( $tags ) );
				?>
			<?php endif; ?>

			<?php
			$comments_number = get_comments_number();
			if ( ! is_single() && $comments_number ) :
				?>
				&nbsp;&nbsp;/&nbsp;&nbsp;
				<a href="<?php comments_link(); ?>">
					<?php
					/* translators: %d is the number of comments */
					printf( esc_html( _n( '%d comment', '%d comments', $comments_number, 'nur' ) ), esc_html( $comments_number ) );
					?>
				</a>
			<?php endif; ?>

		</footer>
		<?php
	}
endif;

if ( ! function_exists( 'nur_render_comment' ) ) :
	/**
	 * Displays a comment.
	 */
	function nur_render_comment( $comment, $args, $depth ) {
		?>
		<li id="comment-<?php comment_ID(); ?>" <?php comment_class( $comment->has_children ? 'parent' : '', $comment ); ?>>
			<article id="div-comment-<?php comment_ID(); ?>" class="comment-body">

				<footer class="comment-footer">

					<div class="comment-author vcard">
						<?php
						if ( 0 !== $args['avatar_size'] ) {
							echo get_avatar( $comment, $args['avatar_size'], '', '', array( 'class' => 'comment-author-avatar' ) );
						}
						echo '<span class="comment-author-name fn">' . get_comment_author_link( $comment ) . '</span>';
						if ( nur_is_post_author( $comment->user_id ) ) {
							echo '<span class="comment-author-label">' . esc_html_x( 'Author', 'comment author label', 'nur' ) . '</span>';
						}
						?>
					</div>

					<div class="comment-date">
						<time datetime="<?php comment_time( 'c' ); ?>">
							<a href="<?php echo esc_url( get_comment_link( $comment, $args ) ); ?>">
								<?php echo esc_html( nur_to_relative_date( $comment->comment_date_gmt ) ); ?>
							</a>
						</time>
					</div>

					<div class="comment-actions">
						<?php
						edit_comment_link( esc_html__( 'Edit', 'nur' ) );

						comment_reply_link( array_merge( $args, array(
							'add_below' => 'div-comment',
							'depth'     => $depth,
						) ) );
						?>
					</div>

					<?php if ( '0' === $comment->comment_approved ) : ?>
						<p class="comment-awaiting-moderation"><?php esc_html_e( 'Your comment is awaiting moderation.', 'nur' ); ?></p>
					<?php endif; ?>

				</footer>

				<div class="comment-content">
					<?php comment_text(); ?>
				</div>

			</article>
		<?php
	}
endif;

if ( ! function_exists( 'nur_header_featured_content' ) ) :
	/**
	 * Displays header featured content with text and icon that were set via the Customizer.
	 */
	function nur_header_featured_content( $id ) {
		$text = get_theme_mod( 'header_featured_content_text_' . $id );
		if ( $text ) {
			$icon_class = get_theme_mod( 'header_featured_content_icon_class_' . $id );
			$icon_style = get_theme_mod( 'header_featured_content_icon_style_' . $id );
			$icon_link = get_theme_mod( 'header_featured_content_icon_link_' . $id );
			?>
			<div class="header-featured-content js-header-featured-content-<?php echo esc_attr( $id ); ?>">

				<?php if ( $icon_class || is_customize_preview() ) : ?>

					<?php if ( $icon_link || is_customize_preview() ) : ?><a href="<?php echo esc_attr( $icon_link ); ?>"><?php endif; ?>

					<div class="header-featured-content-icon <?php if ( $icon_style ) { echo 'header-featured-content-icon--' . esc_attr( $icon_style ); } ?>">
						<span class="<?php echo esc_attr( $icon_class ); ?>"></span>
					</div>

					<?php if ( $icon_link || is_customize_preview() ) : ?></a><?php endif; ?>

				<?php endif; ?>

				<div class="header-featured-content-text">
					<?php echo wp_kses_post( nur_html_entity_decode( $text ) ); ?>
				</div>

			</div>
			<?php
		} elseif ( 1 === $id && is_customize_preview() ) {
			echo '<div class="header-featured-content-placeholder js-header-featured-content-' . esc_attr( $id ) . '"></div>';
		}
	}
endif;

if ( ! function_exists( 'nur_shop_sidebar' ) ) :
	/**
	 * Displays Shop sidebar.
	 */
	function nur_shop_sidebar() {
		if ( ! is_active_sidebar( 'shop-sidebar-1' ) ) {
			return;
		}
		?>

		<div class="sidebar-wrap">
			<aside id="secondary" class="sidebar-content">
				<?php dynamic_sidebar( 'shop-sidebar-1' ); ?>
			</aside>
		</div>

		<?php
	}
endif;
